function stdcell_cutY_far_flush_with_Ti()
% STANDARDIZATION CELL — Cut along Y ONLY on the side far from the origin (Y+)
% Cuts flush with the Ti film (on the upper Y side) only if W > W_Au_top + 2*T_film.

%% ===================== USER PARAMETERS ===============================
% Base (SiO2)
L = 24;     % along X [nm]
W = 10;     % along Y [nm]
H = 2;      % along Z [nm]
origin = [0 0 0];   % [x0 y0 z0] in nm

% Titanium bottom layer (only height variable)
H_Ti_bottom   = 1;  % [nm]

% Gold bottom layer (only height variable)
H_gold_bottom = 2;  % [nm]

% Vacuum (variable X length and Z height; Y = W)
L_vac = 4;          % length in X [nm]
H_vac = 3;          % height in Z [nm]

% Gold top layer (cap, all dimensions variable)
L_Au_top = 16;      % X [nm]
W_Au_top = 8;       % Y [nm]
H_Au_top = 4;       % Z [nm]

% Lateral film around the vertical faces of the top Au (material: Ti)
T_film   = 0.5;     % thickness [nm] (variable)
% ======================================================================

% Plot style
faceColor_base  = [0.55 0.35 0.20]; % brown (SiO2)
faceColor_Ti    = [0.20 0.20 0.20]; % dark grey (Ti)
faceColor_Au    = [1.00 0.84 0.00]; % yellow (Au)
faceColor_vac   = [0.00 0.60 0.00]; % green (vacuum) transparent
alpha_vac       = 0.35;
faceColor_hfo2  = [0.80 0.65 0.45]; % light brown (HfO2) opaque
edgeColor       = 'k';
lineWidth       = 1.0;

% Labels
labelBaseText   = 'SiO2';
labelTiText     = 'Ti';
labelAuText     = 'Au';
labelVacText    = 'vacuum';
labelHfO2Text   = 'HfO2';
labelBaseColor  = [0 0 0];
labelTiColor    = [1 1 1];
labelAuColor    = [0 0 0];
labelVacColor   = [0 0.2 0];
labelHfO2Color  = [0 0 0];
labelSize       = 12;

% ------------------ Local helpers -------------------------------------
vertices_box = @(o,Lx,Wy,Hz) [ o;
                               o + [Lx 0 0];
                               o + [0 Wy 0];
                               o + [0 0 Hz];
                               o + [Lx Wy 0];
                               o + [Lx 0 Hz];
                               o + [0 Wy Hz];
                               o + [Lx Wy Hz] ];
F = [ 1 2 5 3;
      3 5 8 7;
      1 3 7 4;
      2 6 8 5;
      1 4 6 2;
      4 7 8 6 ];

% Crop along Y within [yMinCut, yMaxCut]
    function Vc = cropY(oBox, Lx, Wy, Hz, yMin, yMax)
        y1 = oBox(2); y2 = oBox(2)+Wy;
        if y2 <= yMin || y1 >= yMax
            Vc = [];
            return;
        end
        ya = max(y1, yMin);
        yb = min(y2, yMax);
        Wc = yb - ya;
        if Wc <= 0, Vc = []; else, Vc = vertices_box([oBox(1) ya oBox(3)], Lx, Wc, Hz); end
    end

    function p = drawCroppedY(V, color, varargin)
        if isempty(V), p = []; return; end
        oB  = V(1,:);
        LxB = V(2,1)-V(1,1);
        WyB = V(3,2)-V(1,2);
        HzB = V(4,3)-V(1,3);
        Vc  = cropY(oB, LxB, WyB, HzB, yMinCut, yMaxCut);
        if isempty(Vc), p = []; return; end
        p = patch('Vertices',Vc,'Faces',F,'FaceColor',color, varargin{:});
    end

%% --------------------------- GEOMETRY (uncut) -------------------------
o = origin(:).';
% Base
V_base = vertices_box(o, L, W, H);
% Ti bottom
o_Ti = o + [0 0 H];
V_Ti = vertices_box(o_Ti, L, W, H_Ti_bottom);
% Au bottom
o_Au = o + [0 0 (H + H_Ti_bottom)];
V_Au = vertices_box(o_Au, L, W, H_gold_bottom);

% Top plane (HfO2 | vac | HfO2)
z0_top = o(3) + H + H_Ti_bottom + H_gold_bottom;
L_vac_eff = min(max(L_vac,0), L);
L_left  = max(0, (L - L_vac_eff)/2);
L_right = max(0, L - (L_left + L_vac_eff));

o_left  = [o(1),                 o(2), z0_top];
o_vac   = [o(1) + L_left,        o(2), z0_top];
o_right = [o(1) + L_left + L_vac_eff, o(2), z0_top];

V_left  = vertices_box(o_left,  L_left,    W, H_vac);
V_vac   = vertices_box(o_vac,   L_vac_eff, W, H_vac);
V_right = vertices_box(o_right, L_right,   W, H_vac);

% Centered Au top
x0_cap = o(1) + (L - L_Au_top)/2;
y0_cap = o(2) + (W - W_Au_top)/2;
z0_cap = z0_top + H_vac;
o_cap  = [x0_cap, y0_cap, z0_cap];
V_cap  = vertices_box(o_cap, L_Au_top, W_Au_top, H_Au_top);

% Lateral Ti film
o_film_xL = [x0_cap - T_film,      y0_cap,              z0_cap];
o_film_xR = [x0_cap + L_Au_top,    y0_cap,              z0_cap];
V_film_xL = vertices_box(o_film_xL, T_film,             W_Au_top, H_Au_top);
V_film_xR = vertices_box(o_film_xR, T_film,             W_Au_top, H_Au_top);
o_film_yF = [x0_cap - T_film,      y0_cap - T_film,     z0_cap];
o_film_yB = [x0_cap - T_film,      y0_cap + W_Au_top,   z0_cap];
V_film_yF = vertices_box(o_film_yF, L_Au_top + 2*T_film, T_film,   H_Au_top);
V_film_yB = vertices_box(o_film_yB, L_Au_top + 2*T_film, T_film,   H_Au_top);

% SiO2 frame (filling around film+Au)
x_in_min = x0_cap - T_film;
x_in_max = x0_cap + L_Au_top + T_film;
y_in_min = y0_cap - T_film;
y_in_max = y0_cap + W_Au_top + T_film;

x_base_min = o(1);       x_base_max = o(1) + L;
y_base_min = o(2);       y_base_max = o(2) + W;

T_left_frame  = max(0, x_in_min - x_base_min);
T_right_frame = max(0, x_base_max - x_in_max);
T_front_frame = max(0, y_in_min - y_base_min);
T_back_frame  = max(0, y_base_max - y_in_max);

z0_frame = z0_cap;
H_frame  = H_Au_top;

V_frame_xL = []; V_frame_xR = []; V_frame_yF = []; V_frame_yB = [];
if T_left_frame  > 0, V_frame_xL = vertices_box([x_base_min, y_base_min, z0_frame], T_left_frame,  W, H_frame); end
if T_right_frame > 0, V_frame_xR = vertices_box([x_in_max,   y_base_min, z0_frame], T_right_frame, W, H_frame); end
if T_front_frame > 0, V_frame_yF = vertices_box([x_in_min,   y_base_min, z0_frame], x_in_max - x_in_min, T_front_frame, H_frame); end
if T_back_frame  > 0, V_frame_yB = vertices_box([x_in_min,   y_in_max,   z0_frame], x_in_max - x_in_min, T_back_frame,  H_frame); end

%% -------------------- CUT DEFINITION: ONLY UPPER Y SIDE ---------------
W_pack_ext = W_Au_top + 2*T_film;           % OUTER width (Au+Ti)
if W > W_pack_ext
    % Side near the origin (y0) unchanged; cut only the upper Y side
    yMinCut = o(2);                          
    yMaxCut = y0_cap + W_Au_top + T_film;    % flush with Ti (upper side)
    % safeguard: do not exceed the base
    yMaxCut = min(yMaxCut, o(2)+W);
else
    % No cut
    yMinCut = o(2);                         
    yMaxCut = o(2) + W;                     
end

%% ------------------------------- PLOT ----------------------------------
figure('Color','w'); hold on;

% Base / Ti bottom / Au bottom
p1 = drawCroppedY(V_base, faceColor_base, 'EdgeColor',edgeColor,'LineWidth',lineWidth,'DisplayName','SiO2');
p2 = drawCroppedY(V_Ti,   faceColor_Ti,   'EdgeColor',edgeColor,'LineWidth',lineWidth,'DisplayName','Ti');
p3 = drawCroppedY(V_Au,   faceColor_Au,   'EdgeColor',edgeColor,'LineWidth',lineWidth,'DisplayName','Au');

% HfO2 / Vacuum / HfO2
p4 = drawCroppedY(V_left,  faceColor_hfo2,'EdgeColor',edgeColor,'LineWidth',lineWidth,'DisplayName','HfO2');
p5 = drawCroppedY(V_vac,   faceColor_vac, 'FaceAlpha',alpha_vac,'EdgeColor',edgeColor,'LineWidth',lineWidth,'DisplayName','vacuum');
drawCroppedY(V_right, faceColor_hfo2,'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off');

% Au top + Ti film 
drawCroppedY(V_cap,      faceColor_Au, 'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off');
drawCroppedY(V_film_xL,  faceColor_Ti, 'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off');
drawCroppedY(V_film_xR,  faceColor_Ti, 'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off');
drawCroppedY(V_film_yF,  faceColor_Ti, 'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off');
drawCroppedY(V_film_yB,  faceColor_Ti, 'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off');

% SiO2 frame 
drawCroppedY(V_frame_xL, faceColor_base, 'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off');
drawCroppedY(V_frame_xR, faceColor_base, 'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off');
drawCroppedY(V_frame_yF, faceColor_base, 'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off');
drawCroppedY(V_frame_yB, faceColor_base, 'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off');

% Axes and view
axis equal; grid on; view(35,25);
xlabel('X [nm]'); ylabel('Y [nm]'); zlabel('Z [nm]');
title('Standard Cell — Y cut (only far side) flush with Ti');

% Axis limits with margin
H_tot = H + H_Ti_bottom + H_gold_bottom + H_vac + H_Au_top;
marg  = 0.08 * [L W H_tot];
xlim([o(1)-marg(1), o(1)+L+marg(1)]);
ylim([yMinCut - marg(2), yMaxCut + marg(2)]);
zlim([o(3)-marg(3), o(3)+H_tot+marg(3)]);

% --------- LEGEND (main layers only) -----------------------------------
lh = []; 
if isgraphics(p1), lh(end+1) = p1; end
if isgraphics(p2), lh(end+1) = p2; end
if isgraphics(p3), lh(end+1) = p3; end
if isgraphics(p4), lh(end+1) = p4; end
if ~isempty(p5) && isgraphics(p5), lh(end+1) = p5; end
legend(lh, 'Location','bestoutside');
hold off;

% ------------------------------- LABELS --------------------------------
% (show labels only if the object remains after the crop)
if ~isempty(p1)
    C_base = [o(1)+L/2, (yMinCut+yMaxCut)/2, o(3)+H/2];
    text(C_base(1),C_base(2),C_base(3), labelBaseText,'HorizontalAlignment','center','VerticalAlignment','middle','FontWeight','bold','FontSize',labelSize,'Color',labelBaseColor);
end
if ~isempty(p2)
    C_Ti = [o_Ti(1)+L/2, (yMinCut+yMaxCut)/2, o_Ti(3)+H_Ti_bottom/2];
    text(C_Ti(1),C_Ti(2),C_Ti(3), labelTiText,'HorizontalAlignment','center','VerticalAlignment','middle','FontWeight','bold','FontSize',labelSize,'Color',labelTiColor);
end
if ~isempty(p3)
    C_Au = [o_Au(1)+L/2, (yMinCut+yMaxCut)/2, o_Au(3)+H_gold_bottom/2];
    text(C_Au(1),C_Au(2),C_Au(3), labelAuText,'HorizontalAlignment','center','VerticalAlignment','middle','FontWeight','bold','FontSize',labelSize,'Color',labelAuColor);
end
if ~isempty(p4)
    C_HfO2 = [o(1)+L/2, (yMinCut+yMaxCut)/2, z0_top+H_vac/2];
    text(C_HfO2(1),C_HfO2(2),C_HfO2(3), labelHfO2Text,'HorizontalAlignment','center','VerticalAlignment','middle','FontWeight','bold','FontSize',labelSize,'Color',labelHfO2Color);
end
if ~isempty(p5)
    C_vac = [o(1)+L/2, (yMinCut+yMaxCut)/2, z0_top+H_vac/2];
    text(C_vac(1),C_vac(2),C_vac(3), labelVacText,'HorizontalAlignment','center','VerticalAlignment','middle','FontWeight','bold','FontSize',labelSize,'Color',labelVacColor);
end
end
